// Zeitmessung fr den Overhead des Marshallings bei COM (.NET+CCW)
// Verglichen werden identische, weitgehend leere Implementationen
// - einer Delphi-internen Klasse,
// - einer .NET-Komponente, die per COM Callable Wrappter als
//   COM-Automatisierungsobjekt (IDispatch) erscheint, einmal mit frher,
//   einmal mit spter Bindung.
// - derselben Komponente mit frher und spter Bindung,
//   aber als out-of-process-Server

unit CSComServTestU;

interface

uses
  Windows, Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, CSServerLib_TLB, ComObj, ActiveX;

type
  TAutoEarlyCreator = function: CSServer of object;
  TAutoLateCreator = function: Variant of object;
  TCSComServTestForm = class(TForm)
    bDelphiDirect: TButton;  // Delphi-interne Klasse
    bComEarlybind: TButton;
    bAutoLatebind: TButton;
    bAutoInEarly: TButton;
    Memo1: TMemo;
    Label1: TLabel;
    Label2: TLabel;
    bAutoOutEarly: TButton;
    bAutoOutLate: TButton;
    Label3: TLabel;
    procedure bComEarlybindClick(Sender: TObject);
    procedure bAutoInLateClick(Sender: TObject);
    procedure bAutoInEarlyClick(Sender: TObject);
    procedure bDelphiDirectClick(Sender: TObject);
    procedure bAutoOutEarlyClick(Sender: TObject);
    procedure bAutoOutLateClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    function HostActive: Boolean;
  private
    procedure AutoEarlyBindBench(COMCreator: TAutoEarlyCreator; DivFac: Integer);
    function CreateAutoEarlyInProc: CSServer;
    function CreateAutoEarlyOutProc: CSServer;
    procedure AutoLateBindBench(COMCreator: TAutoLateCreator; DivFac: Integer);
    function CreateAutoLateInProc: Variant;
    function CreateAutoLateOutProc: Variant;
    procedure StartCSComServHost;

  end;

  // Die als COM-Objekt und Automatisierungsobjekt implementierte Klasse
  // ein drittes Mal, hier aber direkt. Die beiden anderen Klassen sind in
  // DelphiComServ und DelphiAutoServ (jeweils _TLB.pas bzw. U.pas)
  TDelphiClass = class(TObject)
  private
    IntVal: Integer;
    StrVal: WideString;
  protected
    function IntParam(Param1: Integer): HResult; stdcall;
    function NoParam: HResult; stdcall;
    function Get_IntProp(out Value: Integer): HResult; stdcall;
    function Get_StrProp(out Value: WideString): HResult; stdcall;
    function StrParam(const Param1: WideString): HResult; stdcall;
  end;



var
  CSComServTestForm: TCSComServTestForm;

implementation
{$R *.dfm}
uses MMSystem;

const Msgs: Array[0..6] of String =
  ('Empty loop', 'NoParam', 'IntParam', 'StrParam', 'StrParam2', 'IntProp', 'StrProp');

// Implementation der Delphi-internen Klasse
function TDelphiClass.IntParam(Param1: Integer): HResult;
begin
  IntVal := Param1; Result := 0;
end;

function TDelphiClass.NoParam: HResult;
begin
  Result := 0;
end;

function TDelphiClass.Get_IntProp(out Value: Integer): HResult;
begin
  Value := IntVal; Result := 0;
end;

function TDelphiClass.Get_StrProp(out Value: WideString): HResult;
begin
  Value := StrVal; Result := 0;
end;

function TDelphiClass.StrParam(const Param1: WideString): HResult;
begin
  StrVal := Param1; Result := 0;
end;
// ---------------------------------------------

procedure TCSComServTestForm.bDelphiDirectClick(Sender: TObject);
var CT: TDelphiClass;
    x, Dummy, BNum: Integer;
    SomeStr: WideString;
    STime: Cardinal;
begin
  Memo1.Lines.Add('*** Delphi-internal class ***');
  STime := timeGetTime;
  for x := 1 to 100000 do
  begin
    CT := TDelphiClass.Create;
    CT.Free;
  end;
  Memo1.Lines.Add(Format('Create (100K): %d',[timeGetTime - STime]));
  CT := TDelphiClass.Create;
  for BNum := 0 to 6 do
  begin
     Dummy := 0;
     STime := timeGetTime;
     for x := 1 to 1000000 do // 1E6
       case BNum of
         0: Inc(Dummy, 1);
         1: Inc(Dummy, CT.NoParam);
         2: Inc(Dummy, CT.IntParam(4));
         3: Inc(Dummy, CT.StrParam('Some string'));
         4: Inc(Dummy, CT.StrParam('Some String with twice as much characters'));
         5: Inc(Dummy, CT.Get_IntProp(Dummy));
         6: Inc(Dummy, CT.Get_StrProp(SomeStr));
       end;
     Memo1.Lines.Add(Msgs[BNum] + ': ' + IntToStr(timeGetTime-STime));
  end;
end;

procedure TCSComServTestForm.bComEarlybindClick(Sender: TObject);
var CT: CSServer;
    x, Dummy, BNum: Integer;
    SomeStr: WideString;
    STime: Cardinal;
begin
  Memo1.Lines.Add('*** COM, early binding ***');
  STime := timeGetTime;
  CT := CoCSServer.Create;

  Memo1.Lines.Add(Format('Create (single): %d',[timeGetTime - STime]));
  x := CT.ProcessID;

  if x = GetCurrentProcessID then Memo1.Lines.Add('in-process')
   else Memo1.Lines.Add('out-of-process');

  STime := timeGetTime;
  for x := 1 to 100000 do
  begin
    CT := CoCSServer.Create;
    CT := nil;
  end;
  Memo1.Lines.Add(Format('Create (100K): %d',[timeGetTime - STime]));


  CT := CoCSServer.Create;
  for BNum := 0 to 6 do
  begin
     Dummy := 0;
     STime := timeGetTime;
     for x := 1 to 1000000 do // 1E6
       case BNum of
         0: Inc(Dummy, 1);
         1: CT.NoParam;
         2: CT.IntParam(4);
         3: CT.StrParam('Some string');
         4: CT.StrParam('Some String with twice as much characters');
         5: Dummy := CT.IntVal;
         6: SomeStr := CT.StrVal;
       end;

     Memo1.Lines.Add(Msgs[BNum] + ': ' + IntToStr(timeGetTime-STime));
  end;
end;

// ---------------------------------------------------------------
procedure TCSComServTestForm.AutoLateBindBench(COMCreator: TAutoLateCreator;
  DivFac: Integer);
var CT: Variant;
    x, Dummy, BNum: Integer;
    SomeStr: WideString;
    STime: Cardinal;
begin
  Memo1.Lines.Add('*** AUTO, late binding ***');
  STime := timeGetTime;
  CT := COMCreator;

  Memo1.Lines.Add(Format('Create (single): %d',[timeGetTime - STime]));

  if CT.ProcessID = GetCurrentProcessID then Memo1.Lines.Add('in-process')
   else Memo1.Lines.Add('out-of-process');


  STime := timeGetTime;
  for x := 1 to 100000 div DivFac do
  begin
    CT := COMCreator;
    CT := Unassigned;
  end;
  Memo1.Lines.Add(Format('Create (100K): %d',[(timeGetTime - STime) * DivFac]));


  CT := COMCreator;
  for BNum := 0 to 6 do
  begin
     Dummy := 0;
     STime := timeGetTime;
     for x := 1 to 1000000 div DivFac do // 1E6
       case BNum of
         0: ;
         1: CT.NoParam;
         2: CT.IntParam(4);
         3: CT.StrParam('Some string');
         4: CT.StrParam('Some String with twice as much characters');
         5: Dummy := CT.IntVal;
         6: SomeStr := CT.StrVal;
       end;

     Memo1.Lines.Add(Msgs[BNum] + ': ' + IntToStr((timeGetTime-STime) * DivFac));
  end;
end;


function TCSComServTestForm.CreateAutoLateInProc: Variant;
begin
  Result := CreateOleObject('CSServerLib.CSServer');
end;

function TCSComServTestForm.CreateAutoLateOutProc: Variant;
var Unknown: IUnknown;
begin
  try
    Result := GetActiveOleObject('CSServerLib.CSServer');
  except
    try
      StartCSComServHost; // Start ComServHost manually
      // retry
      Result := GetActiveOleObject('CSServerLib.CSServer');
    except
      ShowMessage('Unable to start CSComServHost.exe, please try manually');
      raise;
    end;
  end;
end;

procedure TCSComServTestForm.bAutoOutLateClick(Sender: TObject);
begin
  AutoLateBindBench(CreateAutoLateOutProc, 300);
end;


procedure TCSComServTestForm.bAutoInLateClick(Sender: TObject);
begin
  AutoLateBindBench(CreateAutoLateInProc, 10);
end;
// ---------------------------------

function TCSComServTestForm.CreateAutoEarlyInProc: CSServer;
begin
  Result := CoCSServer.Create;
end;

function TCSComServTestForm.CreateAutoEarlyOutProc: CSServer;
var Unknown: IUnknown; ErrCode: Integer;
begin
  ErrCode := GetActiveObject(CLASS_CSServer, nil, Unknown);
  if (ErrCode <> 0) and not HostActive then
  begin
    StartCSComServHost;
    ErrCode := GetActiveObject(CLASS_CSServer, nil, Unknown);
  end;

  if ErrCode <> 0 then
  begin
    raise Exception.Create(Format('COM Error code $%x',[ErrCode]));
  end;

  // CSServer ist von Delphi interessanterweise mit dem GUID 0 besetzt
  Unknown.QueryInterface(IID__CSServer, Result);
end;


procedure TCSComServTestForm.bAutoInEarlyClick(Sender: TObject);
begin
  AutoEarlyBindBench(CreateAutoEarlyInProc, 1);
end;

procedure TCSComServTestForm.bAutoOutEarlyClick(Sender: TObject);
begin
  AutoEarlyBindBench(CreateAutoEarlyOutProc, 100);
end;


procedure TCSComServTestForm.AutoEarlyBindBench(COMCreator: TAutoEarlyCreator; DivFac: Integer);
var CT: CSServer;
    x, Dummy, BNum: Integer;
    SomeStr: WideString;
    STime: Cardinal;
begin
  Memo1.Lines.Add('*** AUTO, early binding ***');
  STime := timeGetTime;
  CT := COMCreator; // EarlyIn oder EarlyOut, d.h. ROT;

  if CT.ProcessID = GetCurrentProcessID then Memo1.Lines.Add('in-process')
   else Memo1.Lines.Add('out-of-process');


  Memo1.Lines.Add(Format('Create (single): %d',[timeGetTime - STime]));

  STime := timeGetTime;
  for x := 1 to 100000 div DivFac do
  begin
    CT := COMCreator; // s. oben
    CT := nil;
  end;
  Memo1.Lines.Add(Format('Create (100K): %d',[(timeGetTime - STime) * DivFac]));

  CT := COMCreator; // s. oben
  for BNum := 0 to 6 do
  begin
     Dummy := 0;
     STime := timeGetTime;
     for x := 1 to 1000000 div DivFac do // 1E6
       case BNum of
         0: ;
         1: CT.NoParam;
         2: CT.IntParam(4);
         3: CT.StrParam('Some string');
         4: CT.StrParam('Some String with twice as much characters');
         5: Dummy := CT.IntVal;
         6: SomeStr := CT.StrVal;
       end;
     Memo1.Lines.Add(Msgs[BNum] + ': ' + IntToStr((timeGetTime-STime) * DivFac));
  end;

end;


function TCSComServTestForm.HostActive: Boolean;
begin
  Result := FindWindow(nil, 'CSComServHostForm - Active') <> 0;
end;


procedure TCSComServTestForm.StartCSComServHost;
begin
   if not HostActive then
     WinExec(PChar(ExtractFilePath(ParamStr(0))+'CSComServHost.exe'), SW_NORMAL);
end;


procedure TCSComServTestForm.FormDestroy(Sender: TObject);
begin
  if HostActive then
    PostMessage(FindWindow(nil, 'CSComServHostForm - Active'), WM_CLOSE, 0, 0);
end;

procedure TCSComServTestForm.FormCreate(Sender: TObject);
begin
  Memo1.Clear;
end;

end.
